
/*
	A lot of useful information about structure of EXE files were found at those
	web-pages:

	1) https://wiki.osdev.org/MZ
	2) https://www.delorie.com/djgpp/doc/exe/
	3) https://habr.com/ru/articles/939506/

	Thank you, guys, for your contribution in spreading knowledge to other
	ones. Without you, we would have lived in stone age.
*/


#include <stdio.h>


#define IMAGE_FILE_EXECUTABLE_IMAGE 0x0002
#define IMAGE_FILE_LINE_NUMS_STRIPPED 0x0004
#define IMAGE_FILE_LOCAL_SYMS_STRIPPED 0x0008
#define IMAGE_FILE_32BIT_MACHINE 0x0100
#define IMAGE_FILE_DEBUG_STRIPPED 0x0200


#define IMAGE_SCN_CNT_CODE 0x00000020 // Section contains code.
#define IMAGE_SCN_CNT_INITIALIZED_DATA       0x00000040  // Section contains initialized data.
#define IMAGE_SCN_MEM_EXECUTE 0x20000000  // Section is executable.
#define IMAGE_SCN_MEM_READ 0x40000000  // Section is readable.
#define IMAGE_SCN_MEM_WRITE                  0x80000000  // Section is writeable.



/*
	Structure of DOS application. ??????????????????????????????????????????????

	---------------------------------
	| Program segment prefix (PSP). |
	---------------------------------
	| DOS header.                   |
	---------------------------------
	| Program image.                |
	---------------------------------

	PSP has fixed size of 256 bytes.

	Size of the header is calculated as a number of header's paragraphs
	multiplied by 16. Number of paragraphs is taken from header's structure.

	Size of the program image is calculated as:
	image_size = 





The offset of the beginning of the EXE data is computed like this:

*/


/*
	Structure of DOS header. The size specified in bytes.


	Offset      | Size | Description
	----------------------------------------------------------------------------
	0x00 - 0x01 | 2    | Signature of the EXE file ("MZ").
	0x02 - 0x03 | 2    | Size of the last block in bytes.
	0x04 - 0x05 | 2    | The number of blocks.
	0x06 - 0x07 | 2    | The number of relocations.
	0x08 - 0x09 | 2    | Size of the header in paragraphs.
	0x0A - 0x0B | 2    | The number of paragraphs required for BSS.
	0x0C - 0x0D | 2    | The number of extra paragraphs needed for the program.
	0x0E - 0x0F | 2    | Offset of stack segment (SS) in EXE file.
	0x10 - 0x11 | 2    | Initial value of stack pointer (SP).
	0x12 - 0x13 | 2    | Checksum of words in the EXE file.
	0x14 - 0x15 | 2    | Initial value of instruction pointer (IP).
	0x16 - 0x17 | 2    | Offset of code segment (CS) in EXE file.
	0x18 - 0x19 | 2    | Offset of relocation table in EXE file.
	0x1A - 0x1B | 2    | Overlay number.
	0x1C - 0x23 | 8    | Reserved.
	0x24 - 0x25 | 2    | OEM's identifier.
	0x26 - 0x27 | 2    | OEM's information.
	0x28 - 0x3B | 20   | Reserved.
	0x3C - 0x3F | 4    | Offset of PE signature.

*/
struct DOS_HEADER
{
	//	Signature of a DOS executable file. The first byte in EXE file must be
	//	"M" (0x4D) and the second is "Z" (0x5A).
	//
	unsigned char Signature[2];

	//	The number of bytes in the last block that are actually used. Every
	//	block in EXE file has 512 byte size but the last block can be smaller.
	//	Valid value is beetween 0 and 512. A zero value indicates that the
	//	whole block is used.
	//
	//		For example, EXE contains 1 block which is not fully filled with a
	//		data. There are only first eight paragraphs contains some non-zero
	//		bytes, the other twenty four paragraphs are empty. So it is
	//		advisable to define size of the last block as 128 (8 paragraphs *
	//		16 bytes).
	//
	//		0x0000	4D 5A 70 00 01 00 00 00 04 00 0F 00 FF FF 00 00
	//		0x0010	B8 00 00 00 00 00 00 00 40 00 1A 00 00 00 00 00
	//		0x0020	00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00
	//		0x0030	00 00 00 00 00 00 00 00 00 00 00 00 00 02 00 00
	//		0x0040	BA 10 00 0E 1F B4 09 CD 21 B8 01 4C CD 21 90 90
	//		0x0050	54 68 69 73 20 70 72 6F 67 72 61 6D 20 6D 75 73
	//		0x0060	74 20 62 65 20 72 75 6E 20 75 6E 64 65 72 20 57
	//		0x0070	69 6E 33 32 0D 0A 24 37 00 00 00 00 00 00 00 00
	//		0x0080	00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00
	//		......	00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00
	//		0x01F0	00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00
	//
	unsigned short BytesInLastBlock;

	//	Number of blocks in the EXE file.
	//
	unsigned short NumOfBlocks;

	//	Number of entries in relocation table. Relocation table is a list of
	//	absolute addresses which point to some chunks of code and data in
	//	object or EXE file. Program loader of an operating system requires this
	//	information for properly shifting this memory addresses before running
	//	the EXE file.
	//
	unsigned short NumOfRelocs;

	//	Number of paragraphs taken up by the header. One paragraph is 16 bytes.
	//	It is used by the loader to find where the actual executable data
	//	starts. It may be larger than what the "standard" fields take up, and
	//	you may use it if you want to include your own header metadata, or put
	//	the relocation table there, or use it for any other purpose.
	//
	unsigned short NumOfHeadParags;

	//	The number of paragraphs required by the program for uninitialized data
	//	(BSS). It is not include the PSP and program image. If there is no free
	//	block big enough, the loading of the EXE stops.
	//
	unsigned short ExtraParagsMin;

	//	Maximum number of paragraphs of additional memory. Normally, the OS
	//	reserves all the remaining conventional memory for your program, but
	//	you can limit it with this field. If no free block is big enough,
	//	the biggest one possible is allocated.
	//
	unsigned short ExtraParagsMax;


	/*	If both the required and additional memory are cleared, MS-DOS will
		attempt to load the executable as high as possible in memory. Otherwise,
		the image will be loaded just above the 256-byte PSP structure, in low
		memory.
	*/


	//	Offset of the stack segment in EXE file. This value is added to the
	//	segment the program was loaded at, and the result is used to initialize
	//	the SS register.
	//
	unsigned short SS_Offset;

	//	Initial value of SP register.
	//
	unsigned short SP;

	//	Word checksum. When added to the sum of all other words in the file,
	//	the result should be zero. Usually, this isn't filled in.
	//
	unsigned short Checksum;

	//	Initial IP register value.
	//
	unsigned short IP;

	//	Offset of the code segment in EXE file. This value is added to the
	//	segment the program was loaded at, and the result is used to initialize
	//	the CS register.
	//
	unsigned short CS_Offset;

	//	Offset of the relocation table.
	//
	unsigned short RelocTableOffset;

	//	Overlay number. Overlays are parts of a program which share the same
	//	chunks of memory. Overlaying is used when amount of available memory is
	//	not enough for loading entire program into it. Zero value means that
	//	it's the main program.
	//
	unsigned short OverlayNum;

	//	Non-standard information added by a compiler or a linker. There is
	//	nothing valuable.
	//
	unsigned short Dump1[4];

	//	OEM's identifier and information. Nothing important here.
	//
	unsigned short OEM_id;
	unsigned short OEM_info;

	//	Reserved space. Not used.
	//
	unsigned short Dump2[10];

	//	Offset of PE signature in EXE file.
	//
	unsigned int PE_Offset;
};


//	Info at
//	https://learn.microsoft.com/en-us/previous-versions/ms809762(v=msdn.10)?redirectedfrom=MSDN
//	https://learn.microsoft.com/en-us/windows/win32/debug/pe-format
//
struct COFF_HEADER
{
	//The Signature field viewed as ASCII text is "PE\0\0"
	unsigned char Signature[4];

	//The CPU that this file is intended for. The following CPU IDs are defined:
	// 0x014c - Intel I386 (same ID used for 486 and 586)
	unsigned short Machine;

	//The number of sections in the file.
	unsigned short NumberOfSections;

	//The time that the linker (or compiler for an OBJ file) produced this file. This field holds the number of seconds since December 31st, 1969, at 4:00 P.M.
	unsigned int TimeDateStamp;

	//The file offset of the COFF symbol table. This field is only used in OBJ files and PE files with COFF debug information. PE files support multiple debug formats, so debuggers should refer to the IMAGE_DIRECTORY_ENTRY_DEBUG entry in the data directory (defined later).
	//
	//MS: The file offset of the COFF symbol table, or zero if no COFF symbol table is present. This value should be zero for an image because COFF debugging information is deprecated. 
	//
	unsigned int PointerToSymbolTable;

	//The number of symbols in the COFF symbol table. See above.
	//
	// MS: The number of entries in the symbol table. This data can be used to locate the string table, which immediately follows the symbol table. This value should be zero for an image because COFF debugging information is deprecated.
	//
	unsigned int NumberOfSymbols;

	//The size of an optional header that can follow this structure. In OBJs, the field is 0. In executables, it is the size of the IMAGE_OPTIONAL_HEADER structure that follows this structure.
	unsigned short SizeOfOptionalHeader;

	//Flags with information about the file. Some important fields:
	//0x0001 - There are no relocations in this file
	//0x0002 - File is an executable image (not a OBJ or LIB) 
	//0x0003 - File is a dynamic-link library, not a program 
	unsigned short Characteristics;
};


struct COFF_HEADER_OPT
{
    //
    // Standard fields. (28 bytes)
    //
    unsigned short    Magic;
    unsigned char    MajorLinkerVersion;
    unsigned char    MinorLinkerVersion;
    unsigned int   SizeOfCode;
    unsigned int   SizeOfInitializedData;
    unsigned int   SizeOfUninitializedData;
    unsigned int   AddressOfEntryPoint;
    unsigned int   BaseOfCode;
    unsigned int   BaseOfData;

    //
    // NT additional fields. (68 bytes)
    //
    unsigned int   ImageBase;
    unsigned int   SectionAlignment;
    unsigned int   FileAlignment;
    unsigned short    MajorOperatingSystemVersion;
    unsigned short    MinorOperatingSystemVersion;
    unsigned short    MajorImageVersion;
    unsigned short    MinorImageVersion;
    unsigned short    MajorSubsystemVersion;
    unsigned short    MinorSubsystemVersion;
    unsigned int   Win32VersionValue;
    unsigned int   SizeOfImage;
    unsigned int   SizeOfHeaders;
    unsigned int   CheckSum;
    unsigned short    Subsystem;
    unsigned short    DllCharacteristics;
    unsigned int   SizeOfStackReserve;
    unsigned int   SizeOfStackCommit;
    unsigned int   SizeOfHeapReserve;
    unsigned int   SizeOfHeapCommit;
    unsigned int   LoaderFlags;
    unsigned int   NumberOfRvaAndSizes;

	//	Data directories. (128 bytes)
	//
	struct
	{
	    unsigned int   VirtualAddress;
	    unsigned int   Size;
	} DataDirectory[16];

	//	Total = 224 bytes
	//
};


struct SECTION_TABLE
{
	unsigned char Name[8];

	union
	{
		unsigned int   PhysicalAddress;
        unsigned int   VirtualSize;
    } Misc;

    unsigned int   VirtualAddress;
    unsigned int   SizeOfRawData;
    unsigned int   PointerToRawData;
    unsigned int   PointerToRelocations;
    unsigned int   PointerToLinenumbers;
    unsigned short    NumberOfRelocations;
    unsigned short    NumberOfLinenumbers;
    unsigned int   Characteristics;
};


struct IMPORT_DIRECTORY_TABLE
{
    union
	{
        unsigned int   Characteristics;            // 0 for terminating null import descriptor
        unsigned int   OriginalFirstThunk;         // RVA to original unbound IAT (PIMAGE_THUNK_DATA)
    };

    int   TimeDateStamp;                  // 0 if not bound,
                                            // -1 if bound, and real date\time stamp
                                            //     in IMAGE_DIRECTORY_ENTRY_BOUND_IMPORT (new BIND)
                                            // O.W. date/time stamp of DLL bound to (Old BIND)

    int   ForwarderChain;                 // -1 if no forwarders
    unsigned int   Name;
    unsigned int   FirstThunk;                     // RVA to IAT (if bound this IAT has actual addresses)
};


/*
struct IMPORT_HINT_TABLE
{
	//An index into the export name pointer table. A match is attempted first with this value. If it fails, a binary search is performed on the DLL's export name pointer table. 
	unsigned short Hint;

	//An ASCII string that contains the name to import. This is the string that must be matched to the public name in the DLL. This string is case sensitive and terminated by a null byte.
	char name[];
};
*/


struct DOS_HEADER MZ =
{
	{'M', 'Z'}, // signature "MZ".
	128, // size of the last block.
	1, // one block is quite enough.
	0, // no need for relocations.
	4, // header consist of 4 paragraphs.
	0, // there is no uninitialized data.
	0xFFFF, // use memory without any constraints.
	0, // stack is not used.
	0, // stack is not used.
	0, // do not use checksum.
	0, // start instructions with beginning.
	0, // do not adjust CS's offset.
	0, // no need for relocation table.
	0, // this is the main program, not an overlay.
	{0}, // dump.
	0, // skip OEM's id.
	0, // skip OEM's info.
	{0}, // dump.
	0x00000080 // offset to PE signature.
};


struct COFF_HEADER COFF =
{
	{'P', 'E', '\0', '\0'}, // signature "PE\0\0".
	0x014c, // machine type is I386+.
	3, // <.text>, <.data> and <.idata> sections.
	946717200, // Sat Jan 01 2000 09:00:00 GMT+0000.
	0, // no debug symbol table.
	0, // no debug symbols.
	sizeof(struct COFF_HEADER_OPT), // size of optional header is always the same.
	IMAGE_FILE_EXECUTABLE_IMAGE | IMAGE_FILE_LINE_NUMS_STRIPPED | IMAGE_FILE_LOCAL_SYMS_STRIPPED | IMAGE_FILE_32BIT_MACHINE | IMAGE_FILE_DEBUG_STRIPPED  // file is 32bit executable and there is no debug info.
};


struct COFF_HEADER_OPT COFF_OPT =
{
	0x010B, // magic number (PE32).
	0, // linker's major version.
	1, // linker's minor version.
	0, // size of code section. ??????????????????????????????????????????????????????????????????????
	0, // size of initialized data. ??????????????????????????????????????????????????????????????????
	0, // size of uninitialized data. ????????????????????????????????????????????????????????????????
	0x00001000, // RVA of entry point.
	0x0000, // offset of code section ????????????????????????????????????????????????????????
	0x0000, // offset of data section ????????????????????????????????????????????????????
	0x00400000, // image base address in memory (4 mbyte offset is default for EXE in WinNT).
	0x00001000, // section alignment in memory (4 kbyte is default for x86-32).
	0x00000200, // section alignment in file (512 bytes is minimum value and default).
	5, // required major version of OS.
	0, // required minor version of OS.
	0, // major version of EXE.
	0, // minor version of EXE.
	5, // required major version of subsystem (5.0 stands for Win 2000).
	0, // required minor version of subsystem.
	0, // ?????????????????????????????????????????????????????????????????????????????????
	0x00004000, // size of image (whole file with headers and sections).
	0x00000200, // size of header (DOS stub + PE + section tables).
	0, // checksum is not used.
	2, // subsystem type. 3 - WINDOWS_CUI (console app), 2 - WINDOWS_GUI (window app).
	0, // dll characteristics ?????????????????????????????????????????????????????????
	0x100000, // stack reserve is 1 mbyte.
	0x1000, // stack commit is 4 kbytes (1 page).
	0x100000, // heap reserve is 1 mbyte.
	0x1000, // heap commit is 4 kbytes.
	0, // no loader's debug flags.
	16, // number of entries in DataDirectory (always 16).
	{
		{0}, // IMAGE_DIRECTORY_ENTRY_EXPORT
		{0x00002000, 0} // IMAGE_DIRECTORY_ENTRY_IMPORT

		// ... other directories
	}
};


struct SECTION_TABLE sections[] =
{
	{
		".txt", // code section
		0x00001000, // virtual size
		0x00001000, // virtual address
		0x00000200, // size of raw data
		0x00000200, // pointer to raw data
		0, // pointer to relocations
		0, // pointer to line numbers
		0, // number of relocations
		0, // number of line numbers
		IMAGE_SCN_CNT_CODE | IMAGE_SCN_MEM_EXECUTE | IMAGE_SCN_MEM_READ
	},

	{
		".idata", // table of import
		0x00001000, // virtual size
		0x00002000, // virtual address
		0x00000200, // size of raw data
		0x00000400, // pointer to raw data
		0, // pointer to relocations
		0, // pointer to line numbers
		0, // number of relocations
		0, // number of line numbers
		IMAGE_SCN_CNT_INITIALIZED_DATA | IMAGE_SCN_MEM_READ
	},

	{
		".data", // variables
		0x00001000, // virtual size
		0x00003000, // virtual address
		0x00000200, // size of raw data
		0x00000600, // pointer to raw data
		0, // pointer to relocations
		0, // pointer to line numbers
		0, // number of relocations
		0, // number of line numbers
		IMAGE_SCN_CNT_INITIALIZED_DATA | IMAGE_SCN_MEM_READ | IMAGE_SCN_MEM_WRITE
	}	
};


struct IMPORT_DIRECTORY_TABLE importDirTable[] =
{
	//	kernel32.dll
	//
	{
		0x0000203C, // RVA to an array of pointers to IMAGE_IMPORT_BY_NAME structure.
		0, // build time (is not important)
		0, // forwarder chain ??????????????????????????????????????????????????????????
		0x00002078, // RVA to dll's name
		0x00002068 // RVA to IMAGE_THUNK_DATA union.
	},

	//	user32.dll
	//
	{
		0x00002044, // RVA to an array of pointers to IMAGE_IMPORT_BY_NAME structure.
		0, // build time (is not important)
		0, // forwarder chain ??????????????????????????????????????????????????????????
		0x00002085, // RVA to dll's name
		0x00002070 // RVA to IMAGE_THUNK_DATA union.
	},

	//	the last enrty must be nulled.
	//
	{0}
};


unsigned int importLookupTable[] =
{
	//	kernel32.dll
	//
	0x0000204C,

	//	user32.dll
	//
	0x0000205A
};


FILE *exeFile;


void writeCode(void);
void writeCoffHeader(void);
void writeCoffHeaderOpt(void);
void writeData(void);
void writeDosHeader(void);
void writeDosStub(void);
void writeImportAddressTable(void);
void writeImportDirTable(void);
void writeImportDllNames(void);
void writeImportHintTable(void);
void writeImportLookupTable(void);
void writeSectionTable(void);


//	Main program.
//
int main(void)
{
	exeFile = fopen("my_testing_exe.exe", "wb");

	writeDosHeader();
	writeDosStub();
	writeCoffHeader();
	writeCoffHeaderOpt();
	writeSectionTable();
	writeCode();
	writeImportDirTable();
	writeImportLookupTable();
	writeImportHintTable();
	writeImportAddressTable();
	writeImportDllNames();
	writeData();

	printf("Ok.");

	fclose(exeFile);
	return 0;
}


//	Write code in .txt section.
//
void writeCode(void)
{
	unsigned short word;
	unsigned int dword;
	int i;

	fputc(0x6A, exeFile); // push 0
	fputc(0x00, exeFile);

	fputc(0x68, exeFile); // push 0x00403000;
	dword = 0x00403000;
	fwrite(&dword, 4, 1, exeFile);

	fputc(0x68, exeFile); // push 0x00403017;
	dword = 0x00403017;
	fwrite(&dword, 4, 1, exeFile);

	fputc(0x6A, exeFile); // push 0
	fputc(0x00, exeFile);

	word = 0x15FF; // call [0x00402070]
	fwrite(&word, 2, 1, exeFile);
	dword = 0x00402070;
	fwrite(&dword, 4, 1, exeFile);

	fputc(0x6A, exeFile); // push 0
	fputc(0x00, exeFile);

	word = 0x15FF; // call [0x00402068]
	fwrite(&word, 2, 1, exeFile);
	dword = 0x00402068;
	fwrite(&dword, 4, 1, exeFile);

	//	Fill free space in this section.
	//
	for(i = 0; i < 484; i++)
		fputc(0x00, exeFile);
}


//	Write common part of PE header to output file.
//
void writeCoffHeader(void)
{
	fwrite(&COFF, 1, 4, exeFile);
	fwrite(&COFF.Machine, 2, 2, exeFile);
	fwrite(&COFF.TimeDateStamp, 4, 3, exeFile);
	fwrite(&COFF.SizeOfOptionalHeader, 2, 2, exeFile);
}


//	Write NT part of PE header to output file.
//
void writeCoffHeaderOpt(void)
{
	fwrite(&COFF_OPT, 2, 1, exeFile);
	fwrite(&COFF_OPT.MajorLinkerVersion, 1, 2, exeFile);
	fwrite(&COFF_OPT.SizeOfCode, 4, 9, exeFile);
	fwrite(&COFF_OPT.MajorOperatingSystemVersion, 2, 6, exeFile);
	fwrite(&COFF_OPT.Win32VersionValue, 4, 4, exeFile);
	fwrite(&COFF_OPT.Subsystem, 2, 2, exeFile);
	fwrite(&COFF_OPT.SizeOfStackReserve, 4, 6, exeFile);
	fwrite(&COFF_OPT.DataDirectory, 4, 32, exeFile);
}


void writeData(void)
{
	int i;
	char string1[] = "a simple PE executable";
	char string2[] = "Hello world!";

	fwrite(string1, 1, 23, exeFile);
	fwrite(string2, 1, 13, exeFile);

	//	Fill free space in this section.
	//
	for(i = 0; i < 476; i++)
		fputc(0x00, exeFile);
}


//	Write DOS header (MZ) to output file.
//
void writeDosHeader(void)
{
	fwrite(&MZ, 1, 2, exeFile);
	fwrite(&MZ.BytesInLastBlock, 2, 29, exeFile);
	fwrite(&MZ.PE_Offset, 4, 1, exeFile);
}


//	Write DOS stub to output file. It prints error message in a terminal for
//	DOS users.
//
void writeDosStub(void)
{
	unsigned short word;
	int i;

	word = 0xC88C; // mov ax,cs
	fwrite(&word, 2, 1, exeFile);

	word = 0xD88E; // mov ds,ax
	fwrite(&word, 2, 1, exeFile);

	fputc(0xBA, exeFile); // mov dx,0010h
	word = 0x0010;
	fwrite(&word, 2, 1, exeFile);

	fputc(0xB4, exeFile); // mov ah,09h
	fputc(0x09, exeFile);

	fputc(0xCD, exeFile); // int 21h
	fputc(0x21, exeFile);

	fputc(0xB8, exeFile); // mov ax,4C01h
	word = 0x4C01;
	fwrite(&word, 2, 1, exeFile);

	fputc(0xCD, exeFile); // int 21h
	fputc(0x21, exeFile);

	fputs("\r\nThis program must be run under Win32.\r\n$", exeFile);

	//	Fill free space in the last paragraph.
	//
	for(i = 0; i < 6; i++)
		fputc(0x00, exeFile);
}


//	Import address table is identical to import lookup table.
//
void writeImportAddressTable(void)
{
	writeImportLookupTable();
}


//	Write import directories table to output file.
//
void writeImportDirTable(void)
{
	int i;
	int num = sizeof(importDirTable) / sizeof(struct IMPORT_DIRECTORY_TABLE);

	for(i = 0; i < num; i++)
	{
		fwrite(&importDirTable[i], 4, 5, exeFile);
	}
}


void writeImportDllNames(void)
{
	int i;
	char name1[] = "kernel32.dll";
	char name2[] = "user32.dll";

	fwrite(name1, 1, 13, exeFile);
	fwrite(name2, 1, 11, exeFile);

	//	Fill free space in PE header. Need to implement smart calculation.
	//
	for(i = 0; i < 368; i++)
		fputc(0x00, exeFile);
}


void writeImportHintTable(void)
{
	unsigned short hint = 0;
	char name1[] = "ExitProcess";
	char name2[] = "MessageBoxA";

	// exitprocess
	//
	fwrite(&hint, 2, 1, exeFile);
	fwrite(name1, 1, 12, exeFile);

	// messageboxa
	//
	fwrite(&hint, 2, 1, exeFile);
	fwrite(name2, 1, 12, exeFile);
}


void writeImportLookupTable(void)
{
	int i;
	int num = sizeof(importLookupTable) / sizeof(int);
	int blank = 0;

	for(i = 0; i < num; i++)
	{
		fwrite(&importLookupTable[i], 4, 1, exeFile);
		fwrite(&blank, 4, 1, exeFile);
	}
}


//	Write section table in output file.
//
void writeSectionTable(void)
{
	int i;
	int num = sizeof(sections) / sizeof(struct SECTION_TABLE);

	for(i = 0; i < num; i++)
	{
		fwrite(&sections[i], 1, 8, exeFile);
		fwrite(&sections[i].Misc, 4, 6, exeFile);
		fwrite(&sections[i].NumberOfRelocations, 2, 2, exeFile);
		fwrite(&sections[i].Characteristics, 4, 1, exeFile);
	}


	//	Fill free space in PE header. Need to implement smart calculation.
	//
	for(i = 0; i < 16; i++)
		fputc(0x00, exeFile);
}


















