
/*******************************************************************************
*
*	Copyright (c) 2024 skipper
*	http://www.heroesden.link
*
*	Terms of usage: Light license agreement v1.
*
*******************************************************************************/


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdarg.h>


/*******************************************************************************
*
*	Language constructs.
*
*******************************************************************************/


//	Block of code placed outside of any language construct.
//
#define LC_NONE 0

//	Single-line comment (including opening forward slashes and newline
//	character).
//
#define LC_SL_COMMENT 1

//	Multi-line comment (including opening and closing forward slashes with
//	asterisks).
//
#define LC_ML_COMMENT 2

//	Double quoted string (including double quotes).
//
#define LC_DBL_QUOTED_STR 3

//	Single quoted string (including single quotes).
//
#define LC_SNG_QUOTED_STR 4


/*******************************************************************************
*
*	Function prototypes.
*
*******************************************************************************/


void abortApp(const char * const msg, ...);
void abortCheck(const char * const msg);

//	Helper functions for buffer's cursor (bc).
//
size_t bcGetLineNum(void);
void bcLocatePos(void);
void bcReset(void);

void checkDoubleQuotes(void);
void checkEscapedChars(void);
void checkExcessSpaces(void);
void checkNonPrintChars(void);
void checkSingleQuotes(void);

//void convertCharType(void);
void convertConstStr(void);
void convertSingleLineCom(void);

void debugLangCons(void);

void readSrcFile(const char * const path);
void writeModFile(void);


/*******************************************************************************
*
*	Global variables.
*
*******************************************************************************/


struct
{
	//	Pointer to dynamically allocated memory which contains whole source
	//	file's code and has extra free space for code conversions.
	//
	char *buffer;

	//	Size of the allocated buffer. It is calculated as SIZEOF(SRC_FILE)*2.
	//
	size_t bufSize;

	//	Cursor in the buffer.
	//
	char *cursor;

	//	Position of the cursor. It is one of a language construct
	//	(LC_* macros).
	//
	size_t cursorPos;

	//	Filesystem path of converted file with ".c" extension. In Windows API
	//	the maximum length for a path is 260 characters.
	//
	char dstPath[260];

	//	Sequential number of the source file in the list of command line
	//	arguments.
	//
	size_t index;

	//	Pointer to the beginning (first character) of a language construct in
	//	which buffer's cursor is placed.
	//
	char *lcBegin;

	//	Pointer to the ending (last character) of a language construct in which
	//	buffer's cursor is placed.
	//
	char *lcEnd;

	//	Filesystem path of source file with ".sals" extension. It taken from
	//	command line arguments.
	//
	const char *path;

	//	Size of unmodified source file.
	//
	size_t size;
}
srcFile;


/*******************************************************************************
*
*	Function declarations.
*
*******************************************************************************/


int main(int argc, char *argv[])
{
	int i;


	//	Does the user specified a file for conversion?
	//
	if(argc < 2)
		abortApp("main(): no input files are specified.");

	//	Handle source files from command line arguments.
	//
	for(i = 1; i < argc; i++)
	{
		srcFile.index = i;
		readSrcFile(argv[i]);

		checkNonPrintChars();
		checkEscapedChars();
		checkExcessSpaces();
		checkDoubleQuotes();
		checkSingleQuotes();

		//	The order of convertation is important! Do not change it.
		//
		convertSingleLineCom();
		convertConstStr();

		writeModFile();
		printf("\n#%d: %s is converted.", srcFile.index, srcFile.path);

		//	Clean up.
		//
		free(srcFile.buffer);
		memset(&srcFile, 0, sizeof(srcFile));
	}

	printf("\n");
	return 0;
}


//	Print an error message to the console and abort the application. This
//	function is simplified version of printf() function and it supports only
//	few specifier character without flags, width, precision and modifiers. The
//	maximum length of the error message is 256 characters including terminating
//	NULL (\0).
//
void abortApp(const char * const msg, ...)
{
	char buffer[256];

	char *ch;
	const char *cursor;
	char spec;
	size_t len;

	va_list vl;
	char *va_str;
	char va_char;
	int va_int;

	if(srcFile.index != 0)
	{
		printf("\nError while processing source file #%d (%s).\n",
			srcFile.index, srcFile.path);

		printf("Buffer's cursor is placed at line #%d.\n", bcGetLineNum());
	}

	printf("\n");

	//	CODING ERROR. The error message is too long for the buffer.
	//
	if(strlen(msg) >= sizeof(buffer))
	{
		printf("abortApp(): message is too long.\nMSG: %s\n", msg);
		exit(EXIT_FAILURE);
	}

	cursor = msg;
	ch = strchr(cursor, '%');

	va_start(vl, msg);

	while(ch != NULL)
	{
		spec = *(ch + 1);

		switch(spec)
		{
			//	Escaped percent is found.
			//
			case '%':

				len = ch - cursor + 1;
				strncpy(buffer, cursor, len);
				buffer[len] = '\0';
				printf(buffer);
				break;

			//	A string is found.
			//
			case 's':

				len = ch - cursor;
				if(len != 0)
				{
					strncpy(buffer, cursor, len);
					buffer[len] = '\0';
					printf(buffer);
				}
	
				va_str = va_arg(vl, char *);
				printf("%s", va_str);
				break;

			//	A digit is found.
			//
			case 'd':

				len = ch - cursor;
				if(len != 0)
				{
					strncpy(buffer, cursor, len);
					buffer[len] = '\0';
					printf(buffer);
				}
	
				va_int = va_arg(vl, int);
				printf("%d", va_int);
				break;

			//	A single character is found.
			//
			case 'c':

				len = ch - cursor;
				if(len != 0)
				{
					strncpy(buffer, cursor, len);
					buffer[len] = '\0';
					printf(buffer);
				}
	
				va_char = va_arg(vl, char);
				printf("%c", va_char);
				break;

			//	CODING ERROR. Unknown specifier character is found.
			//
			default:

				printf("%s [Message filling is aborted because unknown '%c' "
					"specifier character was found.]\n", cursor, spec);
	
				exit(EXIT_FAILURE);
				break;
		}

		//	Search next format specifier.
		//
		cursor = ch + 2;
		ch = strchr(cursor, '%');
	}

	va_end(vl);

	printf("%s\n", cursor);

	//	Is there any library's error?
	//
	if(errno != 0)
	{
		printf("ERRNO(%d): ", errno);
		perror(NULL);
	}

	exit(EXIT_FAILURE);
}


//	Get a line number where buffer's cursor is placed now.
//
size_t bcGetLineNum(void)
{
	char *ch;
	size_t line = 1;

	ch = strchr(srcFile.buffer, '\n');

	while(ch != NULL && ch < srcFile.cursor)
	{
		line++;
		ch = strchr(ch + 1, '\n');
	}

	return line;
}


//	Update information about current position of buffer's cursor. The function
//	only changes value of 'srcFile.cursorPos' variable to corresponding
//	LC_* macros.
//
void bcLocatePos(void)
{
	char * const cur = srcFile.cursor;
	const char * const buf = srcFile.buffer;
	size_t * const pos = &srcFile.cursorPos;


	//	Check if the buffer's cursor leaved a comment or string.
	//
	switch(*pos)
	{
		//	Cursor in a single-line comment.
		//
		case LC_SL_COMMENT:

			//	CODING ERROR. Cursor's position could be equal to
			//	LC_SL_COMMENT starting with (buffer + 1) address only. Also
			//	check if the function is called twice.
			//
			if(cur <= buf || cur == srcFile.lcBegin)
				abortApp("bcLocatePos(): invalid cursor position in a "
					"single-line comment.");

			//	Cursor leaved line with SL-comment and pointed to the first
			//	character in the next line (it could be a comment again).
			//
			if(cur - 1 == srcFile.lcEnd)
			{
				srcFile.lcBegin = NULL;
				srcFile.lcEnd = NULL;
				*pos = LC_NONE;
				break;
			}

			//	Cursor pointed to newline character (last character in the
			//	language construct).
			//
			if(*cur == '\n') srcFile.lcEnd = cur;

			return;

		//	Cursor in a multi-line comment.
		//
		case LC_ML_COMMENT:

			//	Coding error. Cursor's position could be equal to
			//	LC_ML_COMMENT starting with (buffer + 1) address only. Also
			//	check if the function is called twice.
			//
			if(cur <= buf || cur == srcFile.lcBegin)
				abortApp("bcLocatePos(): invalid cursor position in a "
					"multi-line comment.");

			//	Cursor leaved ML-comment and pointed to a next character in the
			//	line (it could be a comment again).
			//
			if(cur - 1 == srcFile.lcEnd)
			{
				srcFile.lcBegin = NULL;
				srcFile.lcEnd = NULL;
				*pos = LC_NONE;
				break;
			}

			//	Cursor pointed to the forward slash in closing delimiter of
			//	ML-comment (last character in the language construct).
			//
			if(*cur == '/' && *(cur - 1) == '*') srcFile.lcEnd = cur;

			return;

		//	Cursor is in a quoted string.
		//
		case LC_SNG_QUOTED_STR:
		case LC_DBL_QUOTED_STR:

			//	Coding error. Cursor's position could be equal to
			//	LC_SNG_QUOTED_STR or LC_DBL_QUOTED_STR starting with
			//	(buffer + 1) address only. Also check if the function is
			//	called twice.
			//
			if(cur <= buf || cur == srcFile.lcBegin)
				abortApp("bcLocatePos(): invalid cursor position in a "
					"quoted string.");

			//	Cursor leaved a string and pointed to the next character after
			//	closing quotes.
			//
			if(cur - 1 == srcFile.lcEnd)
			{
				srcFile.lcBegin = NULL;
				srcFile.lcEnd = NULL;
				*pos = LC_NONE;
				break;
			}

			//	Cursor pointed to an escaped quotes or closing quotes.
			//
			if(*cur == *srcFile.lcBegin)
			{
				//	Amount of backslashes preceding closing quotes. There is
				//	could be escaped backslashes and escaped quotes in the
				//	string.
				//
				size_t backslashes = 0;

				const char *ch = cur - 1;


				//	A quoted string is empty.
				//
				if(ch == srcFile.lcBegin)
				{
					srcFile.lcEnd = cur;
					return;
				}

				while(ch > srcFile.lcBegin)
				{
					if(*ch != '\\') break;

					backslashes++;
					ch--;
				}

				if(backslashes == 0 || backslashes % 2 == 0)
				{
					srcFile.lcEnd = cur;
					return;
				}
			}

			return;
	}

	//	Cursor is not in any language contruct or just leaved it.
	//

	//	A single line comment is found.
	//
	if(*cur == '/' && *(cur + 1) == '/')
	{
		srcFile.lcBegin = cur;
		*pos = LC_SL_COMMENT;
		return;
	}

	//	A multiline comment is found.
	//
	if(*cur == '/' && *(cur + 1) == '*')
	{
		srcFile.lcBegin = cur;
		*pos = LC_ML_COMMENT;
		return;
	}

	//	A double quoted string is found.
	//
	if(*cur == '"')
	{
		srcFile.lcBegin = cur;
		*pos = LC_DBL_QUOTED_STR;
		return;
	}

	//	A single quoted string is found.
	//
	if(*cur == '\'')
	{
		srcFile.lcBegin = cur;
		*pos = LC_SNG_QUOTED_STR;
		return;
	}

	//	Include macro is found.
	//
	if(*cur == 'i')
	{
	}
}


//	Point cursor to the first character in the buffer and purge its current
//	position.
//
void bcReset(void)
{
	srcFile.cursor = srcFile.buffer;
	srcFile.cursorPos = LC_NONE;
}


//	Check double-quoted strings. It is cannot be empty and can contain only
//	few types of escape sequences.
//
void checkDoubleQuotes(void)
{
	char *ch;

	bcReset();

	while(*srcFile.cursor != '\0')
	{
		bcLocatePos();

		//	Skip all language constructs except for double quoted strings and
		//	wait until end of the string will be found.
		//
		if(srcFile.cursorPos != LC_DBL_QUOTED_STR || srcFile.lcEnd == NULL)
		{
			srcFile.cursor++;
			continue;
		}

		//	Empty string is found.
		//
		if(srcFile.lcEnd - srcFile.lcBegin == 1)
			abortApp("checkDoubleQuotes(): There is an empty double-quoted "
				"string. It is not allowed to define a string without any "
				"content.");

		//	Search for escaped characters.
		//
		ch = strchr(srcFile.lcBegin + 1, '\\');

		while(ch != NULL && ch < srcFile.lcEnd)
		{
			switch(*(ch + 1))
			{
				case 'n':
				case 't':
				case '"':
				break;

				default:
				srcFile.cursor = ch;
				abortApp("checkDoubleQuotes(): There is a double-quoted "
					"string containing an invalid escaped character "
					"(ASCII-code: %d). It is allowed only \"\\n\", \"\\t\" "
					"and \"\\\"\" escape sequences in the string.", *(ch + 1));
				break;
			}

			ch = strchr(ch + 1, '\\');
		}

		//	Is the string placed at the very beginning of the source code?
		//
		ch = srcFile.lcBegin;

		if(ch == srcFile.buffer)
		{
			srcFile.cursor = srcFile.lcBegin;

			abortApp("checkDoubleQuotes(): There is a double-quoted string "
				"having non suitable position in the source file.");
		}

		//	Check character before opening quotes.
		//
		switch(*(ch - 1))
		{
			//	'(' - beginning of function parameters.
			//	'{' - beginning of a group in a structure.
			//	' ' and '\t' - indentation.
			//
			case '(':
			case '{':
			case ' ':
			case '\t':
			break;

			default:

				srcFile.cursor = srcFile.lcBegin;

				abortApp("checkDoubleQuotes(): There is an invalid character "
					"before double-quoted string. It is allowed only '(', "
					"'{', ' ' and '\\t' just before opening quotes.");

			break;
		}

		//	Check character after closing quotes.
		//
		ch = srcFile.lcEnd + 1;

		switch(*ch)
		{
			//	',' -	separator of function parameters.
			//	')' -	ending of function parameters or
			//			ending of a group in conditional statement.
			//	';' -	ending of statement.
			//	'}' -	ending of a group in a structure.
			//	' ' -	indentation.
			//	'\n' -	separator between several strings.
			//
			case ',':
			case ')':
			case ';':
			case '}':
			case ' ':
			case '\n':
			break;

			default:

				abortApp("checkDoubleQuotes(): There is an invalid character "
					"after double-quoted string. It is allowed only ',', "
					"')', ';', '}', ' ' and '\\n' right after closing "
					"quotes.");
			break;
		}

		//	All is fine, move on.
		//
		srcFile.cursor++;
	}

	//	Does the last string have closing quotes?
	//
	if(srcFile.cursorPos == LC_DBL_QUOTED_STR && srcFile.lcEnd == NULL)
	{
		srcFile.cursor = srcFile.lcBegin;
		abortApp("checkDoubleQuotes(): There is an opening double quotes "
			"without closing pair.");
	}

	bcReset();
}


//	Check existence of backslashes outside of comments and strings. Backslashes
//	(escaped characters) in comments are ignored but in strings are checked via
//	corresponding "check" functions.
//
void checkEscapedChars(void)
{
	bcReset();

	while(*srcFile.cursor != '\0')
	{
		bcLocatePos();

		//	Skip backslashes from comments and strings.
		//
		switch(srcFile.cursorPos)
		{
			case LC_SL_COMMENT:
			case LC_ML_COMMENT:
			case LC_SNG_QUOTED_STR:
			case LC_DBL_QUOTED_STR:

				srcFile.cursor++;
				continue;

			default: break;
		}

		//	Backslash was found.
		//
		if(*srcFile.cursor == '\\')
			abortApp("checkEscapedChars(): There is an escaped character "
				"outside of a comment or string. Backslashes can be used for "
				"escaping only in comments and strings.");

		srcFile.cursor++;
	}

	bcReset();
}


//	Check existence of redundant spaces in the code (excluding comments and
//	strings). There is shouldn't be any space or tabulation in the end of line
//	(before '\n' character) and code shouldn't contain two consecutive spaces.
//
void checkExcessSpaces(void)
{
	char val;

	bcReset();

	while(*srcFile.cursor != '\0')
	{
		bcLocatePos();

		//	Skip strings.
		//
		switch(srcFile.cursorPos)
		{
			case LC_SNG_QUOTED_STR:
			case LC_DBL_QUOTED_STR:

				srcFile.cursor++;
				continue;

			default: break;
		}

		//	End of line (\n) is found. Need to check if it is first character
		//	in the code or not.
		//
		if(*srcFile.cursor == '\n' && srcFile.cursor != srcFile.buffer)
		{
			val = *(srcFile.cursor - 1);

			if(val == ' ' || val == '\t')
				abortApp("checkExcessSpaces(): There is an redundant space "
					"character in the end of line.");
		}

		//	Check double spaces.
		//
		if(*srcFile.cursor == ' ' && *(srcFile.cursor + 1) == ' ')
		{
			abortApp("checkExcessSpaces(): There is two consecutive spaces "
				"in the code.");
		}

		srcFile.cursor++;
	}

	bcReset();
}


//	Check source code of an input file for non printable characters. It is
//	allowed only spaces, horizontal tabulations, newlines and visible printable
//	characters.
//
void checkNonPrintChars(void)
{
	char ch;

	bcReset();

	while(*srcFile.cursor != '\0')
	{
		ch = *srcFile.cursor;

		if(ch > 126 || (ch < 32 && ch != '\t' && ch != '\n'))
			abortApp("checkNonPrintChars(): There is an invalid character in "
				"input file. It is allowed only spaces, horizontal "
				"tabulations, newlines and visible printable ASCII characters "
				"in the source code.");

		srcFile.cursor++;
	}

	bcReset();
}


//	Check usage of single quotes. Plain CHAR data type must contain only single
//	character (not a string) defined inside of pair of opening and closing
//	single quotes.
//
void checkSingleQuotes(void)
{
	size_t length;
	char *ch;

	bcReset();

	while(*srcFile.cursor != '\0')
	{
		bcLocatePos();

		//	Skip all language constructs except for single quoted strings and
		//	wait until end of the string will be found.
		//
		if(srcFile.cursorPos != LC_SNG_QUOTED_STR || srcFile.lcEnd == NULL)
		{
			srcFile.cursor++;
			continue;
		}

		length = srcFile.lcEnd - srcFile.lcBegin;

		//	Empty string is found.
		//
		if(length == 1)
			abortApp("checkSingleQuotes(): there is an empty single quoted "
				"string.");

		//	Does the string contain printable character?
		//
		if(length == 2)
		{
			char code = *(srcFile.lcBegin + 1);

			if(code < 32 || code > 126)
			{
				srcFile.cursor = srcFile.lcBegin;
				abortApp("checkSingleQuotes(): there is a single quoted "
					"string containing a non-printable character.");
			}
		}

		//	There is several characters in the string (not escaped character).
		//
		if(length > 3)
		{
			srcFile.cursor = srcFile.lcBegin;
			abortApp("checkSingleQuotes(): there is a single quoted string "
				"containing too many characters.");
		}

		//	Escaped string is found.
		//
		if(length == 3)
		{
			//	First character must be backslash.
			//
			if(*(srcFile.lcBegin + 1) != '\\')
			{
				srcFile.cursor = srcFile.lcBegin;
				abortApp("checkSingleQuotes(): there is a single quoted "
					"string containing two invalid characters.");
			}

			//	Is second character valid?
			//
			switch(*(srcFile.lcBegin + 2))
			{
				case 'n':
				case '0':
				case '\\':
				case '\'':
				case 't':
				case 'r':
				break;

				default:
				srcFile.cursor = srcFile.lcBegin;
				abortApp("checkSingleQuotes(): there is a single quoted "
					"string containing an invalid escaped character.");
				break;
			}
		}

		//	Is the string placed at the very beginning of the source code?
		//
		ch = srcFile.lcBegin;

		if(ch == srcFile.buffer)
		{
			srcFile.cursor = srcFile.lcBegin;

			abortApp("checkSingleQuotes(): There is a single-quoted string "
				"having non suitable position in the source file.");
		}

		//	Check character before opening quotes.
		//
		switch(*(ch - 1))
		{
			//	'(' -	beginning of function parameters.
			//	'{' -	beginning of a group in a structure.
			//	' ' -	indentation.
			//	'\t' -	indentation.
			//
			case '(':
			case '{':
			case ' ':
			case '\t':
			break;

			default:

				srcFile.cursor = srcFile.lcBegin;

				abortApp("checkSingleQuotes(): There is an invalid character "
					"before single-quoted string. It is allowed only '(', "
					"'{', ' ' and '\\t' just before opening quotes.");

			break;
		}

		//	Check character after closing quotes.
		//
		ch = srcFile.lcEnd + 1;

		switch(*ch)
		{
			//	',' -	separator of function parameters.
			//	')' -	ending of function parameters or
			//			ending of a group in conditional statement.
			//	';' -	ending of statement.
			//	'}' -	ending of a group in a structure.
			//	' ' -	indentation.
			//	':' -	a case condition in switch statement.
			//
			case ',':
			case ')':
			case ';':
			case '}':
			case ' ':
			case ':':
			break;

			default:

				abortApp("checkSingleQuotes(): There is an invalid character "
					"after single-quoted string. It is allowed only ',', "
					"')', ';', '}', ' ' and ':' right after closing quotes.");
			break;
		}

		//	All is fine, move on.
		//
		srcFile.cursor++;
	}

	//	Does the last string have closing quotes?
	//
	if(srcFile.cursorPos == LC_SNG_QUOTED_STR && srcFile.lcEnd == NULL)
	{
		srcFile.cursor = srcFile.lcBegin;
		abortApp("checkSingleQuotes(): there is an opening single quotes "
			"without closing pair.");
	}

	bcReset();
}





//	Change all CHAR data type to TCHAR. It is skips comments and strings.
//
/*
void convertCharType(void)
{
	//	Cursor in the buffer (current character).
	//
	char *ch;

	//	Pointer to the opening quote of a found string.
	//
	char *strStart;

	//	Type of quotes surrounding a found string. It can be single quotes (')
	//	or double quotes (").
	//
	char quoteType;

	//	Is current character placed in a comment?
	//
	char isComment = 0;

	//	Is current character placed in a string?
	//
	char isString = 0;


	ch = srcFile.buffer;

	while(*ch != '\0')
	{
		//	Beginning of a comment is found. There are could be only multi-line
		//	comments because of single-line comments must be already converted
		//	by convertSingleLineCom() function.
		//
		if(isComment == 0 && *ch == '/' && *(ch + 1) == '*')
		{
			isComment = 1;
			ch += 2;
			continue;
		}

		//	The cursor is inside a comment.
		//
		if(isComment == 1)
		{
			//	Ending of the comment is found.
			//
			if(*ch == '*' && *(ch + 1) == '/')
			{
				isComment = 0;
				ch += 2;
			}

			ch++;
			continue;
		}

		//	Beginning of a string is found.
		//
		if(isString == 0 && (*ch == '\'' || *ch == '"'))
		{
			strStart = ch;
			isString = 1;
			quoteType = *ch;

			ch++;
			continue;
		}

		//	Ending of the string is found. Wrap it with _T() macros.
		//
		if(isString == 1 && *ch == quoteType)
		{
			char *strEnd;

			isString = 0;
			quoteType = NULL;

			//	Add free space before string and fill it with "_T(".
			//
			memmove(strStart + 3, strStart, strlen(strStart) + 1);
			memcpy(strStart, "_T(", 3);

			//	Add free space after string and fill it with ")".
			//
			strEnd = ch + 4;
			memmove(strEnd + 1, strEnd, strlen(strEnd) + 1);
			memcpy(strEnd, ")", 1);

			ch = strEnd + 1;
			continue;
		}

		//	Nothing special is found.
		//
		ch++;
	}
}
*/


//	Wrap all constant strings by _T() macro. It is skips comments and
//	include-macros.
//
void convertConstStr(void)
{
	//	Cursor in the buffer (current character).
	//
	char *ch;

	//	Pointer to the opening quote of a found string.
	//
	char *strStart;

	//	Type of quotes surrounding a found string. It can be single quotes (')
	//	or double quotes (").
	//
	char quoteType;

	//	Is current character placed in a comment?
	//
	char isComment = 0;

	//	Is current character placed in a string?
	//
	char isString = 0;


	ch = srcFile.buffer;

	while(*ch != '\0')
	{
		//	Beginning of a comment is found. There are could be only multi-line
		//	comments because of single-line comments must be already converted
		//	by convertSingleLineCom() function.
		//
		if(isComment == 0 && *ch == '/' && *(ch + 1) == '*')
		{
			isComment = 1;
			ch += 2;
			continue;
		}

		//	Cursor is inside a comment.
		//
		if(isComment == 1)
		{
			//	Ending of the comment is found.
			//
			if(*ch == '*' && *(ch + 1) == '/')
			{
				isComment = 0;
				ch += 2;
			}

			ch++;
			continue;
		}

		//	Skip #include directive.
		//
		if(isString == 0 && *ch == '#')
		{
			int result;

			result = strncmp(ch + 1, "include", 7);

			if(result == 0)
			{
				ch += 8;
				ch = strchr(ch, '\n');

				if(ch == NULL) return;
				else ch++;

				continue;
			}

			ch++;
			continue;
		}

		//	Beginning of a string is found.
		//
		if(isString == 0 && (*ch == '\'' || *ch == '"'))
		{
			strStart = ch;
			isString = 1;
			quoteType = *ch;

			ch++;
			continue;
		}

		//	Ending of the string is found. Wrap it with _T() macros.
		//
		if(isString == 1 && *ch == quoteType)
		{
			char *strEnd;

			isString = 0;
			quoteType = NULL;

			//	Add free space before string and fill it with "_T(".
			//
			memmove(strStart + 3, strStart, strlen(strStart) + 1);
			memcpy(strStart, "_T(", 3);

			//	Add free space after string and fill it with ")".
			//
			strEnd = ch + 4;
			memmove(strEnd + 1, strEnd, strlen(strEnd) + 1);
			memcpy(strEnd, ")", 1);

			ch = strEnd + 1;
			continue;
		}

		//	Nothing special is found.
		//
		ch++;
	}
}


//	Convert single-line comments to multi-line style.
//
void convertSingleLineCom(void)
{
	size_t i;

	char *ch;
	char *comEnd;
	char *comStart;
	char *tabStart;

	char isEndingCom = 0;
	char isStartingCom = 1;
	char isComBlank;

	ch = srcFile.buffer;

	while(*ch != '\0')
	{
		//	An ordinary character is found.
		//
		if(*ch != '/' || *(ch + 1) != '/')
		{
			ch++;
			continue;
		}

		//	Single-line comment is found.
		//
		comStart = ch;
		isComBlank = 1;

		//	Go back and find tabs and spaces.
		//
		tabStart = NULL;

		while(ch > 0 && ch > srcFile.buffer)
		{
			ch--;

			//	Tab or Space is found.
			//
			if(*ch == '\t' || *ch == ' ')
			{
				tabStart = ch;
				continue;
			}

			//	The comment is starting not at the beginning of the line but
			//	after some code. For example,
			//
			//	MyMegaFunc(a, b)
			//	{
			//		return a + b;	//	This is very useful function.
			//	}
			//
			if(*ch != '\n')
			{
				isEndingCom = 1;
				break;
			}

			//	Printable character is found.
			//
			break;
		}

		//	Go forward and find end of the comment line. Also determine if
		//	the comment is blank or not.
		//
		ch = comStart + 2;
		while(*ch != '\n' && *ch != '\0')
		{
			if(*ch != ' ' && *ch != '\t') isComBlank = 0;

			ch++;
		}
		comEnd = ch;

		//	Go forward and find next comment with the same indent.
		//
		if(isEndingCom == 0)
		{
			ch++;

			if(tabStart == NULL && (*ch != '/' || *(ch + 1) != '/'))
			{
				isEndingCom = 1;
			}

			if(tabStart != NULL)
			{
				int result;

				result = strncmp(ch, tabStart, comStart - tabStart + 2);

				if(result != 0)
				{
					isEndingCom = 1;
				}
			}
		}

		//	Change opening comment delimiter.
		//
		if(isStartingCom == 1)
		{
			//	The comment is the first in the comment block (change "//" to
			//	"/*").
			//
			*(comStart + 1) = '*';
		}
		else if(isStartingCom == 0 && isEndingCom == 1 && isComBlank == 1)
		{
			//	The comment is blank and last in the comment block
			//	(change "//" to "*/").
			//
			*comStart = '*';
		}
		else
		{
			//	The comment is placed in the middle or ending of the comment
			//	block (change "//" to " *").
			//
			*comStart = ' ';
			*(comStart + 1) = '*';
		}

		//	Add closing comment delimiter.
		//
		if(isEndingCom == 1)
		{
			if(isStartingCom == 1 || isComBlank == 0)
			{
				ch = comEnd + 3;
	
				//	Add free space just after comment. It is needed for closing
				//	comment delimiter.
				//
				memmove(ch, comEnd, strlen(comEnd) + 1);
	
				//	Add " */" to the end of the comment.
				//
				memcpy(comEnd, " */", 3);
			}
			else
			{
				ch = comEnd;
			}

			//	Settings for next comment.
			//
			isStartingCom = 1;
			isEndingCom = 0;
		}
		else
		{
			//	The comment have continuation on next line.
			//
			isStartingCom = 0;
		}
	}
}


//	Write debugging map with identification numbers of all language constructs
//	contained in the current buffer to the file <srcfilename_lc_map.txt>.
//
void debugLangCons(void)
{
	//	Pointer to the debugging file.
	//
	FILE *fileObj;

	char fileName[ sizeof(srcFile.dstPath) ];


	//	CODING ERROR. Is content of a source file was copied to the buffer?
	//	This function can be called only after readSrcFile().
	//
	if(srcFile.buffer == NULL)
		abortApp("debugLangCons(): the buffer is empty.");

	//	Remove '.sals' extension and append '_lc_map.txt' suffix to the source
	//	file path. Don't need to check existence of dot in file extension,
	//	because of it was done by readSrcFile() function already.
	//
	{
		char *dot;

		strcpy(fileName, srcFile.path);

		dot = strrchr(fileName, '.');
		strcpy(dot, "_lc_map.txt");
	}

	//	Create new file.
	//
	fileObj = fopen(fileName, "w");

	if(fileObj == NULL)
		abortApp("debugLangCons(): can't open the debugging file (%s).",
			fileName);

	//	Write debugging data.
	//
	bcReset();
	fprintf(fileObj, "1: ");

	while(*srcFile.cursor != '\0')
	{
		bcLocatePos();

		if(*srcFile.cursor == '\n')
		{
			fprintf(fileObj, "\n%d: ", bcGetLineNum() + 1);
		}
		else
		{
			fprintf(fileObj, "%d ", srcFile.cursorPos);
		}

		srcFile.cursor++;
	}

	//	The debugging file is not need more.
	//
	fclose(fileObj);
}


//	Read a content of a source file and place it in the buffer.
//
void readSrcFile(const char *path)
{
	//	Pointer to the source file.
	//
	FILE *fileObj;

	srcFile.path = path;

	//	Isn't path of source file too long?
	//
	{
		size_t pathLen = strlen(path);

		if(pathLen >= sizeof(srcFile.dstPath))
			abortApp("readSrcFile(): length of the source file's path is too "
				"long (%d characters).", pathLen);
	}

	//	Copy source file path without '.sal' extension to destination file path
	//	and append '.c' to the end.
	//
	{
		char *dot = strrchr(path, '.');
		size_t dotPos;

		if(dot == NULL)
			abortApp("readSrcFile(): there is no dot in source file's path.");


		dotPos = dot - path;
		strncpy(srcFile.dstPath, path, dotPos + 1);
		srcFile.dstPath[dotPos + 1] = 'c';
		srcFile.dstPath[dotPos + 2] = '\0';
	}

	//	Try to open the source file.
	//
	fileObj = fopen(path, "r");

	if(fileObj == NULL)
		abortApp("readSrcFile(): can't open the source file.");

	//	Calculate the size of the source file.
	//
	if(fseek(fileObj, 0, SEEK_END) != 0)
		abortApp("readSrcFile(): can't find end of the source file.");

	srcFile.size = ftell(fileObj);
	if(srcFile.size == -1L)
		abortApp("readSrcFile(): can't calculate the size of the source file.");

	rewind(fileObj);

	//	Allocate memory for the code of the source file. It is double sized
	//	for upcoming conversions.
	//
	{
		size_t memSize = sizeof(char) * srcFile.size * 2;

		srcFile.buffer = (char*) malloc (memSize);

		if(srcFile.buffer == NULL)
			abortApp("readSrcFile(): can't allocate dynamic memory (%d bytes) "
				"for the buffer.", memSize);

		srcFile.bufSize = memSize;
	}

	//	Copy the content of the source file into the buffer.
	//
	{
		size_t result = fread(srcFile.buffer, sizeof(char), srcFile.size,
			fileObj);

		if(result != srcFile.size)
			abortApp("readSrcFile(): can't copy content of the source file "
				"into the buffer.");
	}

	//	The source file is not need more.
	//
	fclose(fileObj);

	//	Move buffer's cursor to the beginning.
	//
	bcReset();
}


//	Write modified file with ".c" extension after code conversions.
//
void writeModFile(void)
{
	//	Pointer to the modified file.
	//
	FILE *fileObj;


	//	Create new file and write modified source code into it.
	//
	fileObj = fopen(srcFile.dstPath, "w");

	if(fileObj == NULL)
		abortApp("writeModFile(): can't open the destination path for writing "
			"(%s).", srcFile.dstPath);

	//	Write data.
	//
	{
		size_t charCount;
		size_t result;

		charCount = strlen(srcFile.buffer);
		result = fwrite(srcFile.buffer, sizeof(char), charCount, fileObj);

		if(result != charCount)
			abortApp("writeModFile(): can't write the buffer to the "
				"destination path (%s).", srcFile.dstPath);
	}

	//	The destination file is not need more.
	//
	fclose(fileObj);
}

